      <?php

declare(strict_types=1);

namespace Money;

use InvalidArgumentException;
use JsonSerializable;

use function assert;
use function is_numeric;
use function preg_match;
use function sprintf;

/**
 * Currency Pair holding a base, a counter currency and a conversion ratio.
 *
 * @see http://en.wikipedia.org/wiki/Currency_pair
 */
final class CurrencyPair implements JsonSerializable
{
    /**
     PK     h.s[      Currencies.phpnu [        Currency;

    /**
     * Currency to convert to.
     */
    private Currency $counterCurrency;

    /** @psalm-var numeric-string */
    private string $conversionRatio;

    /**
     * @psalm-param numeric-string $conversionRatio
     */
    public function __construct(Currency $baseCurrency, Currency $counterCurrency, string $conversionRatio)
    {
        $this->counterCurrency = $counterCurrency;
        $this->baseCurrency    = $baseCurrency;
        $this->conversionRatio = $conversionRatio;
    }

    /**
     * Creates a new Currency Pair based on "EUR/USD 1.2500" form representation.
     *
     * @param string $iso String representation of the form "EUR/USD 1.2500"
    PK     h.s[      CurrencPK     h.s[8=u	  u	  
  Converter.phpnu [        nction createFromIso(string $iso): CurrencyPair
    {
        $currency = '([A-Z]{2,3})';
        $ratio    = '([0-9]*\.?[0-9]+)'; // @see http://www.regular-expressions.info/floatingpoint.html
        $pattern  = '#' . $currency . '/' . $currency . ' ' . $ratio . '#';

        $matches = [];

        if (! preg_match($pattern, $iso, $matches)) {
            throw new InvalidArgumentException(sprintf('Cannot create currency pair from ISO string "%s", format of string is invalid', $iso));
        }

        assert($matches[1] !== '');
        assert($matches[2] !== '');
        assert(is_numeric($matches[3]));

        return new self(new Currency($matches[1]), new Currency($matches[PK     h.s[8=u	  u	  
  Converter.phpnu [        currency.
     */
    public function getCounterCurrency(): Currency
    {
        return $this->counterCurrency;
    }

    /**
     * Returns the base currency.
     */
    public function getBaseCurrency(): Currency
    {
        return $this->baseCurrency;
    }

    /**
     * Returns the conversion ratio.
     *
     * @psalm-return numeric-string
     */
    public function getConversionRatio(): str