<?php declare(strict_types=1);

/*
 * This file is part of the Monolog package.
 *
 * (c) Jordi Boggiano <j.boggiano@seld.be>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Monolog\Handler;

use Monolog\Formatter\FormatterInterface;
use Monolog\Formatter\LineFormatter;
use Monolog\Utils;
use Monolog\LogRecord;
use Monolog\Level;

use function count;
use function headers_list;
use function stripos;

/**
 * Handler sending logs to browser's javascript console with no browser extension required
 *
 * @author Olivier Poitrey <rs@dailymotion.com>
 */
class BrowserConsoleHandler extends AbstractProcessingHandler
{
    protected static bool $initialized = false;

    /** @var LogRecord[] */
    protected static array $records = [];

    protected const FORMAT_HTML = 'html';
    protected const FORMAT_JS = 'js';
    protected const FORMAT_UNKNOWN = 'unknown';

    /**
     * @inheritDoc
     *
     * Formatted output may contain some formatting markers to be transferred to `console.log` using the %c format.
     *
     * Example of formatted string:
     *
     *     You can do [[blue text]]{color: blue} or [[green background]]{background-color: green; color: white}
     */
    protected function getDefaultFormatter(): FormatterInterface
    {
        return new LineFormatter('[[%channel%]]{macro: autolabel} [[%level_name%]]{font-weight: bold} %message%');
    }

    /**
     * @inheritDoc
     */
    protected function write(LogRecord $record): void
    {
        // Accumulate records
        static::$records[] = $record;

        // Register shutdown handler if not already done
        if (!static::$initialized) {
            static::$initialized = true;
            $this->registerShutdownFunction();
        }
    }

    /**
     * Convert records to javascript console commands and send it to the browser.
     * This method is automatically called on PHP shutdown if output is HTML or Javascript.
     */
    public static function send(): void
    {
        $format = static::getResponseFormat();
        if ($format === self::FORMAT_UNKNOWN) {
            return;
        }

        if (count(static::$records) > 0) {
            if ($format === self::FORMAT_HTML) {
                static::writeOutput('<script>' . self::generateScript() . '</script>');
            } else { // js format
                static::writeOutput(self::generateScript());
            }
            static::resetStatic();
        }
    }

    public function close(): void
    {
        self::resetStatic();
    }

    public function reset(): void
    {
        parent::reset();

        self::resetStatic();
    }

    /**
     * Forget all logged records
     */
    public static function resetStatic(): void
    {
        static::$records = [];
    }

    /**
     * Wrapper for register_shutdown_function to allow overriding
     */
    pro Node/NodeInterface.phpnu [        PK       Qs[1Ug  g              +Node/NodeTrait.phpnu [        PK       Xs[Ů    7            k=Contracts/Adapters/Phpunit/HasPrintableTestCaseName.phpnu [        PK       Xs[y      )            ?Contracts/RenderableOnCollisionEditor.phpnu [        PK       Xs[p  p  !            :AContracts/SolutionsRepository.phpnu [        PK       Xs[,̀                  BContracts/RenderlessEditor.phpnu [        PK       Xs[$                  CContracts/RenderlessTrace.phpnu [        PK       Xs[d2  d2  ,            DAdapters/Phpunit/Printers/DefaultPrinter.phpnu [        PK       Xs[~Y    /            UwAdapters/Phpunit/Printers/ReportablePrinter.phpnu [        PK       Xs[-tG  G              szAdapters/Phpunit/Autoload.phpnu [        PK       Xs[Vm                |Adapters/Phpunit/State.phpnu [        PK       Xs[a^>  >  +            ,Adapters/Phpunit/Subscribers/Subscriber.phpnu [        PK       Xs[nEr/  /  D            řAdapters/Phpunit/Subscribers/EnsurePrinterIsRegisteredSubscriber.phpnu [        PK       Xs[~q                 &Adapters/Phpunit/ConfigureIO.phpnu [        PK       Xs[4w    -            ^Adapters/Phpunit/Support/ResultReflection.phpnu [        PK       Xs[F$  $              FAdapters/Phpunit/TestResult.phpnu [        PK       Xs[:_ЧH  H              Adapters/Phpunit/Style.phpnu [        PK       Xs[J     0            =Adapters/Laravel/IgnitionSolutionsRepository.phpnu [        PK       Xs[J_  _  5            AAdapters/Laravel/Exceptions/RequirementsException.phpnu [        PK       Xs[BU>@b  b  8            CAdapters/Laravel/Exceptions/NotSupportedYetException.phpnu [        PK       Xs[SaYE  E  %            mEAdapters/Laravel/ExceptionHandler.phpnu [        PK       Xs[j    -            QAdapters/Laravel/CollisionServiceProvider.phpnu [        PK       Xs[4Ho/  o/  )            MZAdapters/Laravel/Commands/TestCommand.phpnu [        PK       Xs[vS                Adapters/Laravel/Inspector.phpnu [        PK       Xs[)Ø                Exceptions/ShouldNotHappen.phpnu [        PK       Xs[<      $            ݎExceptions/InvalidStyleException.phpnu [        PK       Xs[+{p  p              Exceptions/TestException.phpnu [        PK       Xs[Z                  Exceptions/TestOutcome.phpnu [        PK       Xs[+                Provider.phpnu [        PK       Xs[޾z                Handler.phpnu [        PK       Xs[                  Highlighter.phpnu [        PK       Xs[<@'  '  
            Writer.phpnu [        PK       Xs[}p                {Coverage.phpnu [        PK       Xs[@X                ArgumentFormatter.phpnu [        PK       Xs[z<    1            SolutionsRepositories/NullSolutionsRepository.phpnu [        PK       Xs[^	                ConsoleColor.phpnu [        PK       Zs["7                ,NetworkExceptionInterface.phpnu [        PK       Zs[EuJ  J              /RequestExceptionInterface.phpnu [        PK       Zs[:                  H2ClientExceptionInterface.phpnu [        PK       |as[g@ɼa  a              ?3exception/Exception.phpnu [        PK       |as[56w  w              4exception/RuntimeException.phpnu [        PK       |as[X@  @              6TypeName.phpnu [        PK       |as[o|                (>type/NullType.phpnu [        PK       |as[/_                Atype/CallableType.phpnu [        PK       |as[@5                Rtype/GenericObjectType.phpnu [        PK       |as[V!p  p              Vtype/FalseType.phpnu [        PK       |as[6  6              Ztype/SimpleType.phpnu [        PK       |as[ӆs                 btype/StaticType.phpnu [        PK       |as[-{                htype/ObjectType.phpnu [        PK       |as[2A                yntype/IterableType.phpnu [        PK       |as[?                ttype/UnknownType.phpnu [        PK       |as[                wtype/VoidType.phpnu [        PK       |as[	                ztype/NeverType.phpnu [        PK       |as[i                ~type/UnionType.phpnu [        PK       |as[k  k              ԉtype/TrueType.phpnu [        PK       |as[<
  
              type/IntersectionType.phpnu [        PK       |as[L/                type/MixedType.phpnu [        PK       |as[P^+    
            type/Type.phpnu [        PK       |as[    
            Parameter.phpnu [        PK       |as[@i.  .              +ReflectionMapper.phpnu [        PK       Dbs[!V  V               UploadedFileFactoryInterface.phpnu [        PK       Dbs[BHA    !            CServerRequestFactoryInterface.phpnu [        PK       Dbs[X"  "              3ResponseFactoryInterface.phpnu [        PK       Dbs[DhE  E              UriFactoryInterface.phpnu [        PK       Dbs[yۜ                -StreamFactoryInterface.phpnu [        PK       Dbs[rTX                RequestFactoryInterface.phpnu [        PK       Ebs[%15 15 	            ?index.phpnu [        PK       
os[nB    '            Renderers/IgnitionExceptionRenderer.phpnu [        PK       
os[n1                Renderers/ErrorPageRenderer.p