<?php declare(strict_types=1);

/*
 * This file is part of the Monolog package.
 *
 * (c) Jordi Boggiano <j.boggiano@seld.be>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Monolog\Handler;

use Closure;
use Monolog\Handler\FingersCrossed\ErrorLevelActivationStrategy;
use Monolog\Handler\FingersCrossed\ActivationStrategyInterface;
use Monolog\Level;
use Monolog\Logger;
use Monolog\ResettableInterface;
use Monolog\Formatter\FormatterInterface;
use Psr\Log\LogLevel;
use Monolog\LogRecord;

/**
 * Buffers all records until a certain level is reached
 *
 * The advantage of this approach is that you don't get any clutter in your log files.
 * Only requests which actually trigger an error (or whatever your actionLevel is) will be
 * in the logs, but they will contain all records, not only those above the level threshold.
 *
 * You can then have a passthruLevel as well which means that at the end of the request,
 * even if it did not get activated, it will still send through log records of e.g. at least a
 * warning level.
 *
 * You can find the various activation strategies in the
 * Monolog\Handler\FingersCrossed\ namespace.
 *
 * @author Jordi Boggiano <j.boggiano@seld.be>
 */
class FingersCrossedHandler extends Handler implements ProcessableHandlerInterface, ResettableInterface, FormattableHandlerInterface
{
    use ProcessableHandlerTrait;

    /**
     * Handler or factory Closure($record, $this)
     *
     * @phpstan-var (Closure(LogRecord|null, HandlerInterface): HandlerInterface)|HandlerInterface
     */
    protected Closure|HandlerInterface $handler;

    protected ActivationStrategyInterface $activationStrategy;

    protected bool $buffering = true;

    protected int $bufferSize;

    /** @var LogRecord[] */
    protected array $buffer = [];

    protected bool $stopBuffering;

    protected Level|null $passthruLevel = null;

    protected bool $bubble;

    /**
     * @phpstan-param (Closure(LogRecord|null, HandlerInterface): HandlerInterface)|HandlerInterface $handler
     *
     * @param Closure|HandlerInterface                    $handler            Handler or factory Closure($record|null, $fingersCrossedHandler).
     * @param int|string|Level|LogLevel::*      $activationStrategy Strategy which determines when this handler takes action, or a level name/value at which the handler is activated
     * @param int                                         $bufferSize         How many entries should be buffered at most, beyond that the oldest items are removed from the buffer.
     * @param bool                                        $bubble             Whether the messages that are handled can bubble up the stack or not
     * @param bool                                        $stopBuffering      Whether the handler should stop buffering after being triggered (default true)
     * @param int|string|Level|LogLevel::*|null $passthruLevel      Minimum level to always flush to handler on close, even if strategy not triggered
     *
     * @phpstan-param value-of<Level::VALUES>|value-of<Level::NAMES>|Level|LogLevel::*|ActivationStrategyInterface $activationStrategy
     * @phpstan-param value-of<Level::VALUES>|value-of<Level::NAMES>|Level|LogLevel::* $passthruLevel
     */
    public function __construct(Closure|HandlerInterface $handler, int|string|Level|ActivationStrategyInterface $activationStrategy = null, int $bufferSize = 0, bool $bubble = true, bool $stopBuffering = true, int|string|Level|null $passthruLevel = null)
    {
        if (null === $activationStrategy) {
            $activationStrategy = new ErrorLevelActivationStrategy(Level::Warning);
        }

        // convert simple int activationStrategy to an object
        if (!$activationStrategy instanceof ActivationStrategyInterface) {
            $activationStr      PK       qHs[A      5             Illuminate/Mail/resources/views/text/button.blade.phpnu [        PK       qHs[(~      5            W Illuminate/Mail/resources/views/text/footer.blade.phpnu [        PK       qHs[yf    6             Illuminate/Mail/resources/views/text/message.blade.phpnu [        PK       qHs[(~      6            
 Illuminate/Mail/resources/views/text/subcopy.blade.phpnu [        PK       qHs[U                ( Illuminate/Mail/composer.jsonnu [        PK       qHs[>e  e  )            P Illuminate/Mail/Events/MessageSending.phpnu [        PK       rHs[^BW  W  &             Illuminate/Mail/Events/MessageSent.phpnu [        PK       rHs[|9  9               Illuminate/Mail/Markdown.phpnu [        PK       rHs[IU&s'  s'              @. Illuminate/Mail/Message.phpnu [        PK       rHs[α3  3              U Illuminate/Mail/LICENSE.mdnu [        PK       rHs[&#    '            {Z Illuminate/Mail/MailServiceProvider.phpnu [        PK       rHs[~6@  6@              a Illuminate/Mail/MailManager.phpnu [        PK       rHs[oMة                < Illuminate/Mail/Attachment.phpnu [        PK       rHs[NO    (            3 Illuminate/Auth/EloquentUserProvider.phpnu [        PK       rHs[IE{Z  Z  .            E Illuminate/Auth/Console/ClearResetsCommand.phpnu [        PK       rHs[1`#
  #
  9             Illuminate/Auth/Console/stubs/make/views/layouts/app.stubnu [        PK       sHs[lTJ  J                Illuminate/Auth/RequestGuard.phpnu [        PK       sHs[JJ  J  +            # Illuminate/Auth/AuthenticationException.phpnu [        PK       sHs[rl7    #             Illuminate/Auth/MustVerifyEmail.phpnu [        PK       sHs[yU)
  )
                Illuminate/Auth/GuardHelpers.phpnu [        PK       sHs[!  !              g Illuminate/Auth/AuthManager.phpnu [        PK       sHs[ 9V  V               Illuminate/Auth/Recaller.phpnu [        PK       sHs[i  i               \% Illuminate/Auth/SessionGuard.phpnu [        PK       tHs[,4Z    ?             Illuminate/Auth/Listeners/SendEmailVerificationNotification.phpnu [        PK       tHs[a@e  e  (             Illuminate/Auth/DatabaseUserProvider.phpnu [        PK       tHs[Ge.	  	  (            Ф Illuminate/Auth/CreatesUserProviders.phpnu [        PK       tHs[Is
  s
              Ү Illuminate/Auth/TokenGuard.phpnu [        PK       tHs[=m/                 Illuminate/Auth/composer.jsonnu [        PK       tHs[+I  I                Illuminate/Auth/Events/Login.phpnu [        PK       tHs[n    "            k Illuminate/Auth/Events/Lockout.phpnu [        PK       uHs[Vd    $            i Illuminate/Auth/Events/Validated.phpnu [        PK       uHs[w:    %            h Illuminate/Auth/Events/Attempting.phpnu [        PK       uHs[~     %             Illuminate/Auth/Events/Registered.phpnu [        PK       uHs["(    .             Illuminate/Auth/Events/CurrentDeviceLogout.phpnu [        PK       uHs[ӻ0̎    (             Illuminate/Auth/Events/Authenticated.phpnu [        PK       uHs[L@    (             Illuminate/Auth/Events/PasswordReset.phpnu [        PK       uHs[y    !             Illuminate/Auth/Events/Logout.phpnu [        PK       uHs[^X    #             Illuminate/Auth/Events/Verified.phpnu [        PK       uHs[ri=  =  !             Illuminate/Auth/Events/Failed.phpnu [        PK       vHs[=    ,            y Illuminate/Auth/Events/OtherDeviceLogout.phpnu [        PK       vHs[.=  =  5            g Illuminate/Auth/Passwords/DatabaseTokenRepository.phpnu [        PK       vHs[KVp_  _  3            	 Illuminate/Auth/Passwords/PasswordBrokerManager.ph