<?php declare(strict_types=1);

/*
 * This file is part of the Monolog package.
 *
 * (c) Jordi Boggiano <j.boggiano@seld.be>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Monolog\Handler;

use Monolog\Level;
use Monolog\Utils;
use Monolog\Formatter\NormalizerFormatter;
use Monolog\Formatter\FormatterInterface;
use Monolog\LogRecord;

/**
 * Class to record a log on a NewRelic application.
 * Enabling New Relic High Security mode may prevent capture of useful information.
 *
 * This handler requires a NormalizerFormatter to function and expects an array in $record->formatted
 *
 * @see https://docs.newrelic.com/docs/agents/php-agent
 * @see https://docs.newrelic.com/docs/accounts-partnerships/accounts/security/high-security
 */
class NewRelicHandler extends AbstractProcessingHandler
{
    /**
     * @inheritDoc
     */
    public function __construct(
        int|string|Level $level = Level::Error,
        bool $bubble = true,

        /**
         * Name of the New Relic application that will receive logs from this handler.
         */
        protected string|null $appName = null,

        /**
         * Some context and extra data is passed into the handler as arrays of values. Do we send them as is
         * (useful if we are using the API), or explode them for display on the NewRelic RPM website?
         */
        protected bool $explodeArrays = false,

        /**
         * Name of the current transaction
         */
        protected string|null $transactionName = null
    ) {
        parent::__construct($level, $bubble);
    }

    /**
     * @inheritDoc
     */
    protected function write(LogRecord $record): void
    {
        if (!$this->isNewRelicEnabled()) {
            throw new MissingExtensionException('The newrelic PHP extension is required to use the NewRelicHandler');
        }

        if (null !== ($appName = $this->getAppName($record->context))) {
            $this->setNewRelicAppName($appName);
        }

        if (null !== ($transactionName = $this->getTransactionName($record->context))) {
            $this->setNewRelicTransactionName($transactionName);
            unset($record->formatted['context']['transaction_name']);
        }

        if (isset($record->context['exception']) && $record->context['exception'] instanceof \Throwable) {
            newrelic_notice_error($record->message, $record->context['exception']);
            unset($record->formatted['context']['exception']);
        } else {
            newrelic_notice_error($record->message);
        }

        if (isset($record->formatted['context']) && is_array($record->formatted['context'])) {
            foreach ($record->formatted['context'] as $key => $parameter) {
                if (is_array($parameter) && $this->explodeArrays) {
                    foreach ($parameter as $paramKey => $paramValue) {
                        $this->setNewRelicParameter('context_' . $key . '_' . $paramKey, $paramValue);
                    }
                } else {
                    $this->setNewRelicParameter('context_' . $key, $parameter);
                }
            }
        }

        if (isset($record->formatted['extra']) && is_array($record->formatted['extra'])) {
            foreach ($record->formatted['extra'] as $key => $parameter) {
                if (is_array($parameter) && $this->explodeArrays) {
                    foreach ($parameter as $paramKey => $paramValue) {
                        $this->setNewRelicParameter('extra_' . $key . '_' . $paramKey, $paramValue);
                    }
                } else {
                    $this->setNewRelicParameter('extra_' . $key, $parameter);
                }
            }
        }
    }

    /**
     * Checks whether the NewRelic extension is enabled in the system.
     *	    1            w" Illuminate/Support/Traits/CapsuleManagerTrait.phpnu [        PK       Hs[	&!c  c  &            o( Illuminate/Support/Traits/Tappable.phpnu [        PK       Hs[-8j	  	  .            (* Illuminate/Support/Traits/ReflectsClosures.phpnu [        PK       Hs[]Ju  u  )            d4 Illuminate/Support/Traits/Localizable.phpnu [        PK       Hs[α3  3              27 Illuminate/Support/LICENSE.mdnu [        PK       Hs[;+!  !  !            ; Illuminate/Support/HtmlString.phpnu [        PK       Hs[y\  \  !            $@ Illuminate/Support/Pluralizer.phpnu [        PK       Hs[뷴^(  ^(  !            K Illuminate/Support/MessageBag.phpnu [        PK       Hs[    #            t Illuminate/Support/ProcessUtils.phpnu [        PK       Hs[p                | Illuminate/Support/Fluent.phpnu [        PK       Hs[;oֶ                  Illuminate/Support/Benchmark.phpnu [        PK       Hs[!N                 Illuminate/Support/Timebox.phpnu [        PK       Hs[cv    /             Illuminate/Support/AggregateServiceProvider.phpnu [        PK       Hs[	3    5            Ƞ Illuminate/Support/Testing/Fakes/PendingChainFake.phpnu [        PK       Hs[!  !  5            Ǧ Illuminate/Support/Testing/Fakes/PendingBatchFake.phpnu [        PK       Hs[$Z;B  B  .            M Illuminate/Support/Testing/Fakes/BatchFake.phpnu [        PK       Hs[?Lr^  r^  ,             Illuminate/Support/Testing/Fakes/BusFake.phpnu [        PK       Hs[G@4  @4  -             Illuminate/Support/Testing/Fakes/MailFake.phpnu [        PK       Hs[Ar9  9  .            XN Illuminate/Support/Testing/Fakes/QueueFake.phpnu [