<?php declare(strict_types=1);

/*
 * This file is part of the Monolog package.
 *
 * (c) Jordi Boggiano <j.boggiano@seld.be>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Monolog\Handler;

use Monolog\Formatter\FormatterInterface;
use Monolog\Level;
use Monolog\Utils;
use Monolog\Handler\Slack\SlackRecord;
use Monolog\LogRecord;

/**
 * Sends notifications through Slack API
 *
 * @author Greg Kedzierski <greg@gregkedzierski.com>
 * @see    https://api.slack.com/
 */
class SlackHandler extends SocketHandler
{
    /**
     * Slack API token
     */
    private string $token;

    /**
     * Instance of the SlackRecord util class preparing data for Slack API.
     */
    private SlackRecord $slackRecord;

    /**
     * @param  string                    $token                  Slack API token
     * @param  string                    $channel                Slack channel (encoded ID or name)
     * @param  string|null               $username               Name of a bot
     * @param  bool                      $useAttachment          Whether the message should be added to Slack as attachment (plain text otherwise)
     * @param  string|null               $iconEmoji              The emoji name to use (or null)
     * @param  bool                      $useShortAttachment     Whether the context/extra messages added to Slack as attachments are in a short style
     * @param  bool                      $includeContextAndExtra Whether the attachment should include context and extra data
     * @param  string[]                  $excludeFields          Dot separated list of fields to exclude from slack message. E.g. ['context.field1', 'extra.field2']
     * @throws MissingExtensionException If no OpenSSL PHP extension configured
     */
    public function __construct(
        string $token,
        string $channel,
        ?string $username = null,
        bool $useAttachment = true,
        ?string $iconEmoji = null,
        $level = Level::Critical,
        bool $bubble = true,
        bool $useShortAttachment = false,
        bool $includeContextAndExtra = false,
        array $excludeFields = [],
        bool $persistent = false,
        float $timeout = 0.0,
        float $writingTimeout = 10.0,
        ?float $connectionTimeout = null,
        ?int $chunkSize = null
    ) {
        if (!extension_loaded('openssl')) {
            throw new MissingExtensionException('The OpenSSL PHP extension is required to use the SlackHandler');
        }

        parent::__construct(
            'ssl://slack.com:443',
            $level,
            $bubble,
            $persistent,
            $timeout,
            $writingTimeout,
            $connectionTimeout,
            $chunkSize
        );

        $this->slackRecord = new SlackRecord(
            $channel,
            $username,
            $useAttachment,
            $iconEmoji,
            $useShortAttachment,
            $includeContextAndExtra,
            $excludeFields
        );

        $this->token = $token;
    }

    public function getSlackRecord(): SlackRecord
    {
        return $this->slackRecord;
    }

    public function getToken(): string
    {
        return $this->token;
    }

    /**
     * @inheritDoc
     */
    protected function generateDataStream(LogRecord $record): string
    {
        $content = $this->buildContent($rec      Fs[Vs  s              x autoload.phpnu [        PK       dGs[r
  r
              D| Utils/ArrayList.phpnu [        PK       dGs[T|!  |!               Utils/Reflection.phpnu [        PK       dGs[>;6k  k               Utils/DateTime.phpnu [        PK       dGs[_                f Utils/FileInfo.phpnu [        PK       dGs[ L|V  |V               Utils/Strings.phpnu [        PK       dGs[ԇ`?
  
              p Utils/Callback.phpnu [        PK       dGs[ڏ,  ,               Utils/Arrays.phpnu [        PK       dGs[,Ө)  )              K Utils/Validators.phpnu [        PK       dGs[9 
   
              su Utils/Helpers.phpnu [        PK       dGs[h64  64               Utils/Finder.phpnu [        PK       dGs[4+                * Utils/Json.phpnu [        PK       dGs[)0!L  !L              J Utils/Html.phpnu [        PK       dGs[OId  d              	 Utils/ObjectHelpers.phpnu [        PK       dGs[X>,  ,              T% Utils/ReflectionMethod.phpnu [        PK       dGs[
S_VJ$  J$              ( Utils/FileSystem.phpnu [        PK       dGs[ع$                XM Utils/ImageType.phpnu [        PK       dGs[ani  i              @O Utils/ImageColor.phpnu [        PK       dGs[H                U Utils/Paginator.phpnu [        PK       dGs[d                g Utils/exceptions.phpnu [        PK       dGs[fH  H              k Utils/Floats.phpnu [        PK       dGs[[ K  K              s Utils/Random.phpnu [        PK       dGs[﷽Y^  ^              x Utils/Image.phpnu [        PK       dGs[q  q              ֹ Utils/ArrayHash.phpnu [        PK       dGs[1l  l              ޹ Utils/Type.phpnu [        PK       dGs[6O  O              9 HtmlStringable.phpnu [        PK       dGs[5\N,  ,               StaticClass.phpnu [        PK       dGs[                5 exceptions.phpnu [        PK       dGs[,Y  Y              > Iterators/Mapper.phpnu [        PK       dGs[V                 Iterators/CachingIterator.phpnu [        PK       dGs[s)=
  
               SmartObject.phpnu [        PK       dGs[rk                ," compatibility.phpnu [        PK       dGs[A                % Translator.phpnu [        PK       IHs[ݓ&  &  )            ' Illuminate/Queue/QueueServiceProvider.phpnu [        PK       IHs[,8+  +              N Illuminate/Queue/Queue.phpnu [        PK       IHs[H$(  (  ,            y Illuminate/Queue/InvalidPayloadException.phpnu [        PK       IHs[    $            | Illuminate/Queue/ListenerOptions.phpnu [        PK       IHs[KX8    /             Illuminate/Queue/Console/FailedTableCommand.phpnu [        PK       IHs[U}"  "  *             Illuminate/Queue/Console/ListenCommand.phpnu [        PK       IHs[ 	  	  +            j Illuminate/Queue/Console/RestartCommand.phpnu [        PK       IHs[;S
  
  )            Ι Illuminate/Queue/Console/ClearCommand.phpnu [        PK       IHs[є    .             Illuminate/Queue/Console/ListFailedCommand.phpnu [        PK       IHs[p  p  0             Illuminate/Queue/Console