<?php declare(strict_types=1);

/*
 * This file is part of the Monolog package.
 *
 * (c) Jordi Boggiano <j.boggiano@seld.be>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Monolog;

use Psr\Log\LogLevel;

/**
 * Represents the log levels
 *
 * Monolog supports the logging levels described by RFC 5424 {@see https://datatracker.ietf.org/doc/html/rfc5424}
 * but due to BC the severity values used internally are not 0-7.
 *
 * To get the level name/value out of a Level there are several options:
 *
 * - Use ->getName() to get the standard Monolog name which is full uppercased (e.g. "DEBUG")
 * - Use ->toPsrLogLevel() to get the standard PSR-3 name which is full lowercased (e.g. "debug")
 * - Use ->toRFC5424Level() to get the standard RFC 5424 value (e.g. 7 for debug, 0 for emergency)
 * - Use ->name to get the enum case's name which is capitalized (e.g. "Debug")
 *
 * To get the internal value for filtering, if the includes/isLowerThan/isHigherThan methods are
 * not enough, you can use ->value to get the enum case's integer value.
 */
enum Level: int
{
    /**
     * Detailed debug information
     */
    case Debug = 100;

    /**
     * Interesting events
     *
     * Examples: User logs in, SQL logs.
     */
    case Info = 200;

    /**
     * Uncommon events
     */
    case Notice = 250;

    /**
     * Exceptional occurrences that are not errors
     *
     * Examples: Use of deprecated APIs, poor use of an API,
     * undesirable things that are not necessarily wrong.
     */
    case Warning = 300;

    /**
     * Runtime errors
     */
    case Error = 400;

    /**
     * Critical conditions
     *
     * Example: Application component unavailable, unexpected exception.
     */
    case Critical = 500;

    /**
     * Action must be taken immediately
     *
     * Example: Entire website down, database unavailable, etc.
     * This should trigger the SMS alerts and wake you up.
     */
    case Alert = 550;

    /**
     * Urgent alert.
     */
    case Emergency = 600;

    /**
     * @param value-of<self::NAMES>|LogLevel::*|'Debug'|'Info'|'Notice'|'Warning'|'Error'|'Critical'|'Alert'|'Emergency' $name
     * @return static
     */
    public static function fromName(string $name): self
    {
        return match ($name) {
            'debug', 'Debug', 'DEBUG' => self::Debug,
            'info', 'Info', 'INFO' => self::Info,
            'notice', 'Notice', 'NOTICE' => self::Notice,
            'warning', 'Warning', 'WARNING' => self::Warning,
            'error', 'Error', 'ERROR' =>          CSSList/CSSList.phpnu [        PK       MAs[z  z              Q CSSList/AtRuleBlockList.phpnu [        PK       MAs[T                 CSSList/Document.phpnu [        PK       MAs[eHet  t               CSSList/CSSBlockList.phpnu [        PK       MAs[K'  '  
            	 Rule/Rule.phpnu [        PK       :Fs[SR,  ,  &            1 Faker/Container/ContainerException.phpnu [        PK       :Fs[A2/6  6              3 Faker/Container/Container.phpnu [        PK       :Fs[f+  +  &            B Faker/Container/ContainerInterface.phpnu [        PK       ;Fs[d<
  <
  $            D Faker/Container/ContainerBuilder.phpnu [        PK       ;Fs[/  /  +            N Faker/Container/NotInContainerException.phpnu [        PK       ;Fs[                ,P Faker/Calculator/Ean.phpnu [        PK       ;Fs[                T Faker/Calculator/Isbn.phpnu [        PK       ;Fs[󖤒                I[ Faker/Calculator/Luhn.phpnu [        PK       ;Fs[$t!                wb Faker/Calculator/Iban.phpnu [        PK       ;Fs[^9                Xi Faker/Calculator/Inn.phpnu [        PK       ;Fs[l-                n Faker/Calculator/TCNo.phpnu [        PK       ;Fs[:j    &            Rs Faker/ORM/Mandango/EntityPopulator.phpnu [        PK       ;Fs[[w  w                Faker/ORM/Mandango/Populator.phpnu [        PK       ;Fs[O".W  W  (            P Faker/ORM/Mandango/ColumnTypeGuesser.phpnu [        PK       ;Fs[ Q  Q  $             Faker/ORM/Propel/EntityPopulator.phpnu [        PK       <Fs[4"
  
               Faker/ORM/Propel/Populator.phpnu [        PK       <Fs[)Wa
  
  &            ï Faker/ORM/Propel/ColumnTypeGuesser.phpnu [        PK       <Fs[    "             Faker/ORM/Spot/EntityPopulator.phpnu [        PK       <Fs[6Z
  
               Faker/ORM/Spot/Populator.phpnu [        PK       <Fs[#3    $            v Faker/ORM/Spot/ColumnTypeGuesser.phpnu [        PK       <Fs[s    %             Faker/ORM/Propel2/EntityPopulator.phpnu [        PK       <Fs[!1LC  C               Faker/ORM/Propel2/Populator.phpnu [        PK       <Fs[S$
  
  '            W Faker/ORM/Propel2/ColumnTypeGuesser.phpnu [        PK       <Fs[si    &            h Faker/ORM/Doctrine/EntityPopulator.phpnu [        PK       <Fs[
  
               3 Faker/ORM/Doctrine/Populator.phpnu [        PK       <Fs[f9z
  z
  (    