<?php
/**
 * Environment cli file.
 *
 * @package Cli
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 */

namespace App\Cli;

use App\Exceptions\AppException;
use App\Json;
use App\Language;
use App\Process;
use App\Utils\ConfReport;

/**
 * Environment cli class.
 */
class Environment extends Base
{
	/** {@inheritdoc} */
	public string $moduleName = 'Environment';

	/** {@inheritdoc} */
	protected array $methods = [
		'confReportErrors' => 'Server configuration - Errors',
		'confReportAll' => 'Server configuration - All',
	];

	/**
	 * Server configuration - Errors.
	 *
	 * @return void
	 */
	public function confReportErrors(): void
	{
		Process::$requestMode = 'Cron';
		ConfReport::$sapi = 'cron';
		foreach (ConfReport::$types as $type) {
			$table = [];
			foreach (ConfReport::getErrors($type, true) as $name => $item) {
				if (!empty($item['isHtml'])) {
					$this->htmlToText($item);
				}
				$table[] = [
					'Parameter' => $name,
					'Recommended' => $item['recommended'] ?? '-',
					'Value' => $item['cron'] ?? '-',
				];
			}
			if ($table) {
				$this->climate->border('─', 140);
				$this->climate->underline()->bold()->out("		$type");
				$this->climate->border('─', 140);
				$this->climate->table($table);
			}
		}
	}

	/**
	 * Server configuration - All.
	 *
	 * @return void
	 * @throws AppException
	 */
	public function confReportAll(): void
	{
		Process::$requestMode = 'Cron';
		ConfReport::$sapi = 'cron';
		$all = ConfReport::getAll();
		foreach ($all as $type => $items) {
			$table = [];
			foreach ($items as $name => $item) {
				if (!$item['testCli']) {
					continue;
				}
				if (!empty($item['isHtml'])) {
					$this->htmlToText($item);
				}
				$value = $item['cron'] ?? $item['www'] ?? '';
				if (str_starts_with($value, 'LBL_')) {
					$value = Language::translate($value);
				}
				$value = is_array($value) ? Json::encode($value) : $value;
				$table[] = [
					'Parameter' => $item['status'] ? $name : "<light_red>$name</light_red>",
					'Recommended' => empty($item['recommended']) ? '-' : print_r($item['recommended'], true),
					'Value' => $item['status'] ? $value : ("<light_red>$value</light_red>"),
				];
			}
			if ($table) {
				$this->climate->border('─', 140);
				$this->climate->underline()->bold()->out("		$type");
				$this->climate->border('─', 140);
				$this->climate->table($table);
			}
		}
	}

	/**
	 * Convert html to cli text.
	 *
	 * @param array &$item
	 *
	 * @return void
	 */
	public function htmlToText(array &$item): void
	{
		if (str_contains($item['val'], '<b class="text-danger">')) {
			$item['val'] = preg_replace_callback("'<b class=\"text-danger\">(.*?)</b>'si", fn ($match) => "<light_red>{$match['1']}</light_red>", $item['val']);
		}
		if (str_contains($item['recommended'], '<b class="text-danger">')) {
			$item['recommended'] = preg_replace_callback("'<b class=\"text-danger\">(.*?)</b>'si", fn ($match) => "<light_red>{$match['1']}</light_red>", $item['recommended']);
		}
	}
}
