<?php

/**
 * Provider for login with LDAP.
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 * @author Antoni Kiszka <a.kiszka@yetiforce.com>
 */

namespace App\Integrations\UserAuth;

/**
 * LDAP login provider - class.
 */
class LDAP extends \App\Authenticator\Password
{
	/** {@inheritdoc} */
	protected $label = 'LDAP';

	/** {@inheritdoc} */
	protected $icon = 'yfi-ldap';

	/** {@inheritdoc} */
	protected $additionalInfo = 'LBL_LDAP_RECOMMENDED_INFO';

	/** {@inheritdoc} */
	protected $editFieldNames = ['server', 'port', 'domain'];

	/** {@inheritdoc} */
	public function verifyPassword(#[\SensitiveParameter] string &$password, bool $falsify = false): bool
	{
		if ($falsify || !$this->isActive()) {
			return parent::verifyPassword($password, $falsify);
		}

		$pwd = $password;
		$password = '[REDACTED]';

		$port = $this->get('port') ?: 389;
		$server = $this->get('server');
		$domain = $this->get('domain');
		$ds = ldap_connect($server, $port);
		if (!$ds) {
			\App\Log::error('Error LDAP authentication: Could not connect to LDAP server.', 'UserAuthentication');
		}
		ldap_set_option($ds, LDAP_OPT_PROTOCOL_VERSION, 3); // Try version 3.  Will fail and default to v2.
		ldap_set_option($ds, LDAP_OPT_REFERRALS, 0);
		ldap_set_option($ds, LDAP_OPT_TIMELIMIT, 5);
		ldap_set_option($ds, LDAP_OPT_TIMEOUT, 5);
		ldap_set_option($ds, LDAP_OPT_NETWORK_TIMEOUT, 5);
		if ('tls' === parse_url($server)['scheme']) {
			ldap_start_tls($ds);
		}
		$bind = ldap_bind($ds, $this->userModel->getDetail('user_name') . $domain, $pwd);
		if (!$bind) {
			\App\Log::error('LDAP authentication: LDAP bind failed. |' . ldap_errno($ds) . '|' . ldap_error($ds), 'UserAuthentication');
		}

		return $bind;
	}

	/** {@inheritdoc} */
	public function preProcess()
	{
		\App\Session::set('UserAuthMethod', 'LDAP');
	}

	/** {@inheritdoc} */
	public function getFieldInstanceByName(string $name): ?\Vtiger_Field_Model
	{
		$moduleName = 'Settings:UserAuth';
		$field = ['uitype' => 1, 'column' => $name, 'name' => $name, 'displaytype' => 1, 'typeofdata' => 'V~M', 'presence' => 0, 'isEditableReadOnly' => false];
		switch ($name) {
			case 'server':
				$field['label'] = 'LBL_LDAP_SERVER';
				$field['purifyType'] = \App\Purifier::URL;
				$field['maximumlength'] = '100';
				$field['fieldvalue'] = $this->has($name) ? $this->get($name) : '';
				break;
			case 'port':
				$field['label'] = 'LBL_LDAP_PORT';
				$field['typeofdata'] = 'I~M';
				$field['maximumlength'] = '1,65535';
				$field['purifyType'] = \App\Purifier::INTEGER;
				$field['fieldvalue'] = $this->has($name) ? $this->get($name) : '';
				break;
			case 'domain':
				$field['label'] = 'LBL_LDAP_DOMAIN';
				$field['maximumlength'] = '100';
				$field['typeofdata'] = 'V~O';
				$field['purifyType'] = \App\Purifier::URL;
				$field['fieldvalue'] = $this->has($name) ? $this->get($name) : '';
				break;
			default:
				$field = [];
				break;
		}

		return $field ? \Vtiger_Field_Model::init($moduleName, $field, $name) : null;
	}
}
