<?php
/**
 * Class to get coordinates.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Tomasz Kur <t.kur@yetiforce.com>
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 */

namespace App\Map;

use App\Exceptions\AppException;
use yii\db\Exception;

/**
 * Base Connector to get coordinates.
 */
class Coordinates extends BaseConnector
{
	/**
	 * @var string Table name of coordinates for records
	 */
	public const COORDINATES_TABLE_NAME = 'u_#__openstreetmap';

	/**
	 * @var string Table name of updater for records
	 */
	public const UPDATER_TABLE_NAME = 'u_#__openstreetmap_record_updater';

	/**
	 * @var string[] Type of address.
	 */
	public const TYPE_ADDRESS = ['a', 'b', 'c'];
	/**
	 * {@inheritdoc}
	 */
	protected static string $configTableName = 'a_#__map_coordinates_driver_config';

	/**
	 * Function to get base information about address from Vtiger_Record_Model.
	 *
	 * @param \Vtiger_Record_Model $recordModel
	 * @param string               $type
	 *
	 * @return string[]
	 */
	public static function getAddressParams(\Vtiger_Record_Model $recordModel, string $type): array
	{
		$street = $recordModel->get('addresslevel8' . $type) ?: null;
		$buildingNumber = $recordModel->get('buildingnumber' . $type) ?: '';

		return [
			'street' => trim($street ? $street . ' ' . $buildingNumber : ''),
			'post_code' => trim($recordModel->get('addresslevel7' . $type) ?: ''),
			'city' => trim($recordModel->get('addresslevel5' . $type) ?: ''),
			'county' => trim($recordModel->get('addresslevel3' . $type) ?: ''),
			'state' => trim($recordModel->get('addresslevel2' . $type) ?: ''),
			'country' => trim($recordModel->get('addresslevel1' . $type) ?: ''),
		];
	}

	/**
	 * Save map coordinates.
	 *
	 * @param int    $id
	 * @param string $type        Type, @see self::TYPE_ADDRESS
	 * @param array  $coordinates
	 * @param bool   $updater
	 *
	 * @throws Exception
	 *
	 * @return void
	 */
	public static function saveCoordinates(int $id, string $type, array $coordinates, bool $updater = true): void
	{
		$createCommand = \App\Db::getInstance()->createCommand();
		if (
			(new \App\Db\Query())->from(self::COORDINATES_TABLE_NAME)
				->where(['crmid' => $id, 'type' => $type])->exists()
		) {
			if (empty($coordinates['lat']) && empty($coordinates['lon'])) {
				$createCommand->delete(
					self::COORDINATES_TABLE_NAME,
					['crmid' => $id, 'type' => $type]
				)->execute();
			} else {
				$createCommand->update(
					self::COORDINATES_TABLE_NAME,
					[
						'lat' => $coordinates['lat'],
						'lon' => $coordinates['lon'],
					],
					['crmid' => $id, 'type' => $type]
				)->execute();
			}
			if ($updater) {
				$createCommand->delete(self::UPDATER_TABLE_NAME, ['crmid' => $id, 'type' => $type])->execute();
			}
		} else {
			if (!empty($coordinates['lat']) && !empty($coordinates['lon'])) {
				$createCommand->insert(self::COORDINATES_TABLE_NAME, [
					'type' => $type,
					'crmid' => $id,
					'lat' => $coordinates['lat'],
					'lon' => $coordinates['lon'],
				])->execute();
				if ($updater) {
					$createCommand->delete(self::UPDATER_TABLE_NAME, ['crmid' => $id, 'type' => $type])->execute();
				}
			}
		}
	}

	/**
	 * Save map record updater.
	 *
	 * @param int    $id
	 * @param string $type
	 * @param array  $address
	 *
	 * @throws Exception
	 * @throws AppException
	 *
	 * @return void
	 */
	public static function saveUpdater(int $id, string $type, array $address): void
	{
		$isCoordinateExists = (new \App\Db\Query())
			->from(self::UPDATER_TABLE_NAME)
			->where(['crmid' => $id, 'type' => $type])
			->exists();

		if (!$isCoordinateExists) {
			\App\Db::getInstance()->createCommand()
				->insert(self::UPDATER_TABLE_NAME, [
					'crmid' => $id,
					'type' => $type,
					'address' => \App\Json::encode($address),
				])->execute();
		} else {
			\App\Db::getInstance()->createCommand()
				->update(
					self::UPDATER_TABLE_NAME,
					['address' => \App\Json::encode($address)],
					['crmid' => $id, 'type' => $type]
				)->execute();
		}
	}

	/**
	 * Deletes coordinates from 2 tables.
	 *
	 * @param int    $id
	 * @param string $type
	 *
	 * @throws Exception
	 * @throws \Throwable
	 *
	 * @return void
	 */
	public static function deleteAll(int $id, string $type): void
	{
		$db = \App\Db::getInstance();

		$transaction = $db->beginTransaction();
		try {
			$command = $db->createCommand();
			$command->delete(
				self::UPDATER_TABLE_NAME,
				[
					'crmid' => $id,
					'type' => $type,
				]
			)->execute();

			$command->delete(
				self::COORDINATES_TABLE_NAME,
				[
					'crmid' => $id,
					'type' => $type,
				]
			)->execute();

			$transaction->commit();
		} catch (\Throwable $e) {
			$transaction->rollBack();

			throw $e;
		}
	}

	/**
	 * {@inheritdoc}
	 *
	 * @throws AppException
	 */
	public static function getDriverClass(string $driver): string
	{
		$className = "\\App\\Map\\Coordinates\\{$driver}";
		if (!class_exists($className)) {
			throw new AppException('ERR_CLASS_NOT_FOUND|' . $driver);
		}

		return $className;
	}
}
