<?php
/**
 * Base Connector to find routing between two points.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 */

namespace App\Map\Routing;

use App\Map\BaseDriver;

/**
 * Base Connector to get routing.
 */
abstract class Base extends BaseDriver
{
	/**
	 * Documentation URL.
	 *
	 * @var string
	 */
	protected string $docUrl;

	/** @var float[] */
	protected array $start;

	/** @var float[] */
	protected array $end;

	/** @var array */
	protected array $indirectPoints;

	/** @var string Rout description */
	protected string $description = '';

	/** @var float Travel time */
	protected float $travelTime = 0;

	/** @var float */
	protected float $distance = 0;

	/** @var array GeoJSON, route geometry format */
	protected array $geoJson;

	/** @var array Return details */
	protected array $returnDetails = [
		'navigationInstruction' => false,
		'duration' => false,
		'polyline' => false,
	];

	/**
	 * Function to calculate routing.
	 */
	abstract public function calculate(): void;

	/**
	 * Function to parse points.
	 *
	 * @return array
	 */
	abstract public function parsePoints(): array;

	/**
	 * Get documentation URL.
	 *
	 * @return string
	 */
	public function getDocUrl(): string
	{
		return $this->docUrl;
	}

	/**
	 * Set return detail.
	 *
	 * @param string $name
	 * @param bool   $value
	 *
	 * @return void
	 */
	public function setReturnDetail(string $name, bool $value): void
	{
		$this->returnDetails[$name] = $value;
	}

	/**
	 * Set start of routing.
	 *
	 * @param float $lat
	 * @param float $lon
	 */
	public function setStart(float $lat, float $lon): void
	{
		$this->start = ['lat' => $lat, 'lon' => $lon];
	}

	/**
	 * Set end of routing.
	 *
	 * @param float $lat .
	 * @param float $lon
	 */
	public function setEnd(float $lat, float $lon): void
	{
		$this->end = ['lat' => $lat, 'lon' => $lon];
	}

	/**
	 * Add indirect point.
	 *
	 * @param float $lat
	 * @param float $lon
	 */
	public function addIndirectPoint(float $lat, float $lon): void
	{
		$this->indirectPoints[] = ['lat' => $lat, 'lon' => $lon];
	}

	/**
	 * Get geojson - RFC 7946.
	 *
	 * @return array
	 */
	public function getGeoJson(): array
	{
		return $this->geoJson;
	}

	/**
	 * Get description.
	 *
	 * @return string
	 */
	public function getDescription(): string
	{
		return $this->description;
	}

	/**
	 * Get time of travel.
	 *
	 * @return float
	 */
	public function getTravelTime(): float
	{
		return $this->travelTime;
	}

	/**
	 * Get distance of routing.
	 *
	 * @return float
	 */
	public function getDistance(): float
	{
		return $this->distance;
	}
}
