<?php

/**
 * Sanitize fields.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Leszek Koziatek <l.koziatek@yetiforce.com>
 * @author Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Report\Sanitizer;

use App\FieldCoordinatorTransformer\QueryGeneratorFieldTransformer;
use App\Fields\Currency;
use App\Fields\Owner;
use App\Module;
use App\Purifier;
use App\Record;
use App\Report\Factory\FieldFactory;
use App\Report\Model\Expression;
use App\Report\Model\Field;
use App\Report\Model\Query;

/** ReportFieldSanitizer class */
class ReportFieldSanitizer implements ReportSanitizer
{
	public function __construct(
		private readonly FieldFactory $fieldFactory,
	) {}

	/** {@inheritDoc} */
	public function sanitize(Query $query, array $data = []): array
	{
		$result = [];

		foreach ($data as $i => $row) {
			$result[$i] = [];
			foreach ($row as $fieldCoordinate => $value) {
				try {
					$field = $this->fieldFactory->create($query, $fieldCoordinate);
					$result[$i][$fieldCoordinate] = $this->convert($field, $value);
				} catch (\Throwable $throwable) {
					$result[$i][$fieldCoordinate] = $this->getNonFieldValue($fieldCoordinate, $value, $query);
				}
			}
		}

		return $result;
	}

	/** Get display data for expression not related with Yetiforce fields */
	protected function getNonFieldValue(string $fieldCoordinate, mixed $value, Query $query): mixed
	{
		$fieldParts = QueryGeneratorFieldTransformer::extract($fieldCoordinate);

		$modulePK = '';
		$fieldName = $fieldParts['fieldName'];

		if (null !== $moduleName = $fieldParts['moduleName']) {
			$moduleName = preg_replace('/_.*/', '', $moduleName);

			if (true === Module::isModuleActive($moduleName)) {
				$moduleModel = \Vtiger_Module_Model::getInstance($moduleName);
				$modulePK = $moduleModel->basetableid;
			}
		}

		if(('id' === $fieldName || $modulePK === $fieldName) && false === $this->isFunctionField($query, $fieldParts)){
			$value = 'Users' === $moduleName ? Owner::getUserLabel($value) : Record::getLabel($value);
		}

		return $value;
	}

	/** Check if field is related with any functions */
	protected function isFunctionField(Query $query, array $fieldParts): bool
	{
		/** @var array<Expression> $relatedExpression */
		$relatedExpression = array_values(array_filter(
			$query->getExpressions(),
			static fn (Expression $expression): bool => QueryGeneratorFieldTransformer::extract($expression->getAlias()) === $fieldParts,
		));

		return \count($relatedExpression) > 0 && null !== $relatedExpression[0]->getFunction();
	}

	/** Get data in format prepared for display data */
	private function convert(Field $fieldModel, mixed $value): float|int|string
	{
		/** @var \Vtiger_Field_Model|\Vtiger_Basic_InventoryField $nativeModel */
		$nativeModel = $fieldModel->getNativeModel();

		return match ($nativeModel->getFieldDataType()) {
			'date', 'datetime' => false === is_numeric($value) ? $nativeModel->getDisplayValue($value) : $value,
			'phone' => $nativeModel->getDisplayValue($value, rawText:true),
			'text' => trim(Purifier::purify(preg_replace('/\\s+/', ' ', $value))),
			'image', 'multiImage' => '',
			'categoryMultipicklist' => null === $value ? '' : $nativeModel->getDisplayValue($value, rawText:true),
			'inventory' => match ($nativeModel->getType()) {
				'Currency' => Currency::getById($value)['currency_code'] ?? '',
				default => $nativeModel->getDisplayValue($value, rawText: true),
			},
			default => $nativeModel->getDisplayValue($value, rawText:true)
		};
	}
}
