<?php

/**
 * Sanitize fields.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Report\Sanitizer;

use App\Fields\Currency;
use App\Purifier;
use App\Report\Factory\FieldFactory;
use App\Report\Model\Field;
use App\Report\Model\Query;

/** XlsReportFieldSanitizer class */
final class XlsReportFieldSanitizer extends ReportFieldSanitizer
{
	public function __construct(
		private readonly FieldFactory $fieldFactory,
	) {}

	/** {@inheritDoc} */
	public function sanitize(Query $query, array $data = []): array
	{
		$result = [];

		foreach ($data as $i => $row) {
			$result[$i] = [];
			foreach ($row as $fieldCoordinate => $value) {
				try {
					$field = $this->fieldFactory->create($query, $fieldCoordinate);
					$result[$i][$fieldCoordinate] = $this->convert($field, $value);
				} catch (\Throwable $th) {
					$result[$i][$fieldCoordinate] = $this->getNonFieldValue($fieldCoordinate, $value, $query);
				}
			}
		}

		return $result;
	}

	/** Get data in format prepared for xls file */
	private function convert(Field $fieldModel, mixed $value): float|int|string
	{
		/** @var \Vtiger_Field_Model|\Vtiger_Basic_InventoryField $nativeModel */
		$nativeModel = $fieldModel->getNativeModel();

		return match ($nativeModel->getFieldDataType()) {
			'integer', 'double' => $nativeModel->getDisplayValue($value, recordModel: false, rawText: true),
			'currency', 'currencyInventory' => number_format((float) $value, 2, '.', ''),
			'date', 'datetime' => false === is_numeric($value) ? $nativeModel->getDisplayValue($value) : $value,
			'phone' => $nativeModel->getDisplayValue($value, false, false, true),
			'text' => trim(Purifier::purify(preg_replace('/\\s+/', ' ', $value))),
			'image', 'multiImage' => '',
			'categoryMultipicklist' => null === $value ? '' : $nativeModel->getDisplayValue($value, false, false, true),
			'inventory' => $this->parseInventoryData($nativeModel, $value),
			default => ($nativeModel instanceof \Vtiger_Basic_InventoryField)
				? $nativeModel->getDisplayValue($value, [], true)
				: $nativeModel->getDisplayValue($value, false, false, true),
		};
	}

	/** Parse data from inventory field */
	private function parseInventoryData(\Vtiger_Basic_InventoryField $fieldType, $value): int|string
	{
		return match ($fieldType->getType()) {
			'Margin', 'TotalPrice' => number_format((float) $value, 2, '.', ''),
			'Currency' => Currency::getById($value)['currency_code'],
			'Date', 'Name', 'Reference', 'Value', 'Unit', 'Boolean', 'Comment', 'Picklist', 'PicklistField',
			'DiscountMode', 'TaxMode' => $fieldType->getDisplayValue($value, rawText: true),
			default => $fieldType->getDisplayValue($value, rawText: true),
		};
	}
}
