<?php

/**
 * Create condition template row.
 *
 * @package   ReportTemplates
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Paweł Karczmarcyk <p.karczmarczyk@yetiforce.com>
 */

use App\Controller\ClearProcess;
use App\Controller\ExposeMethod;
use App\Controller\View\Page;
use App\Exceptions\AppException;
use App\Exceptions\NoPermitted;
use App\Purifier;
use App\Request;

class ReportTemplate_Conditions_View extends Page
{
	use ClearProcess;
	use ExposeMethod;

	/** {@inheritdoc} */
	public function __construct()
	{
		parent::__construct();
		$this->exposeMethod('row');
		$this->exposeMethod('builder');
	}

	/** {@inheritdoc} */
	public function checkPermission(Request $request): void
	{
		if (!Users_Privileges_Model::getCurrentUserPrivilegesModel()->hasModulePermission($request->getModule())) {
			throw new NoPermitted('ERR_PERMISSION_DENIED', 406);
		}
	}

	/** {@inheritdoc}
	 * @throws AppException
	 */
	public function process(Request $request): void
	{
		if ($mode = $request->getMode()) {
			$this->invokeExposedMethod($mode, $request);
		} else {
			$this->row($request);
		}
	}

	/**
	 * Display one condition for a field.
	 *
	 * @param Request $request
	 *
	 * @throws AppException
	 *
	 * @return void
	 */
	public function row(Request $request): void
	{
		$baseModule = $request->getByType('baseModule', Purifier::ALNUM);
		$relatedModules = $request->getArray('relatedModules', Purifier::INTEGER);
		$relatedModuleSkip = $request->getBoolean('relatedModuleSkip', false);
		$sourceModuleModel = Vtiger_Module_Model::getInstance($baseModule);

		$fieldInfo = false;
		if ($request->isEmpty('fieldname')) {
			$fieldModel = current($sourceModuleModel->getFields());
		} else {
			$fieldInfo = $request->getForSql('fieldname', false);
			[$fieldName, $fieldModuleName, $sourceFieldName] = array_pad(explode(':', $fieldInfo), 3, false);
			if ('INVENTORY' === $sourceFieldName) {
				$fieldModel = $sourceModuleModel->getInventoryModel()->getField($fieldName);
			} elseif (!empty($sourceFieldName)) {
				$fieldModel = Vtiger_Field_Model::getInstance($fieldName, Vtiger_Module_Model::getInstance($fieldModuleName));
			} else {
				$fieldModel = Vtiger_Field_Model::getInstance($fieldName, $sourceModuleModel);
				if ('virtual' === $fieldModel->getFieldDataType()) {
					$fieldModel = $fieldModel->getUITypeModel()->getVirtualFieldModel();
				}
			}
		}
		$operators = $request->isEmpty('parent', 1) ? $fieldModel->getQueryOperatorLabels() : $fieldModel->getRecordOperatorLabels();
		if ($request->isEmpty('operator', true)) {
			$selectedOperator = key($operators);
		} else {
			$selectedOperator = $request->getByType('operator', Purifier::ALNUM);
		}

		$viewer = $this->getViewer($request);
		$viewer->assign('OPERATORS', $operators);
		$viewer->assign('SELECTED_OPERATOR', $selectedOperator);
		$viewer->assign('SELECTED_FIELD_MODEL', $fieldModel);
		$viewer->assign('RECORD_STRUCTURE_RELATED_MODULES', !$relatedModuleSkip ? $this->getRelatedStructures($baseModule) : []);
		$viewer->assign('RECORD_STRUCTURE_RELATIONS_MODULES', $relatedModules ? $this->getRelationsModules($sourceModuleModel, $relatedModules) : []);
		$viewer->assign('RECORD_STRUCTURE', Vtiger_RecordStructure_Model::getInstanceForModule($sourceModuleModel)->getStructure());
		$viewer->assign('RECORD_STRUCTURE_INV', $sourceModuleModel->isInventory() ? $this->getInvStructure($baseModule) : []);
		$viewer->assign('FIELD_INFO', $fieldInfo);
		$viewer->assign('SOURCE_MODULE', $baseModule);
		$viewer->view('ConditionBuilderRow.tpl', $request->getModule());
	}

	/**
	 * @param Vtiger_Module_Model $moduleModel
	 *
	 * @throws AppException
	 *
	 * @return array
	 */
	public function inventoryFields(Vtiger_Module_Model $moduleModel): array
	{
		$inventory = [];
		foreach ($moduleModel->getInventoryModel()->getFields() as $item) {
			$inventory[] = $item->getName();
		}
		return $inventory;
	}

	/**
	 * Display the condition builder panel.
	 *
	 * @param Request $request
	 *
	 * @return void
	 */
	public function builder(Request $request): void
	{
		$baseModule = $request->getByType('baseModule', Purifier::ALNUM);
		$relatedModules = $request->getArray('relatedModules', Purifier::INTEGER);

		$advanceCriteria = $request->getArray('advanceCriteria', Purifier::TEXT);
		$relatedModuleSkip = $request->getBoolean('relatedModuleSkip', false);

		$sourceModuleModel = Vtiger_Module_Model::getInstance($baseModule);
		$viewer = $this->getViewer($request);
		if ($record = $request->getInteger('record')) {
			$viewer->assign('RECORD', Vtiger_Record_Model::getInstanceById($record, 'ReportTemplate'));
		}
		$viewer->assign('ADVANCE_CRITERIA', $advanceCriteria);
		$viewer->assign('RECORD_STRUCTURE_RELATED_MODULES', !$relatedModuleSkip ? $this->getRelatedStructures($baseModule) : []);
		$viewer->assign('RECORD_STRUCTURE_RELATIONS_MODULES', $relatedModules ? $this->getRelationsModules($sourceModuleModel, $relatedModules) : []);
		$viewer->assign('RECORD_STRUCTURE', Vtiger_RecordStructure_Model::getInstanceForModule($sourceModuleModel)->getStructure());
		$viewer->assign('RECORD_STRUCTURE_INV', $sourceModuleModel->isInventory() ? $this->getInvStructure($baseModule) : []);
		$viewer->assign('SOURCE_MODULE', $baseModule);
		$viewer->view('ConditionBuilder.tpl', $request->getModule());
	}

	/**
	 * @param Vtiger_Module_Model $moduleModel
	 * @param array               $relatedModules
	 *
	 * @return array
	 */
	private function getRelationsModules(Vtiger_Module_Model $moduleModel, array $relatedModules): array
	{
		$block = [];
		foreach ($relatedModules as $item) {
			$module = $moduleModel->getRelations()[$item];

			$block[$module->getRelationModuleName()] = [
				\sprintf('%s:M2M', $item) => $module->getRelationModuleModel()->getFieldsByBlocks()
			];
		}
		return $block;
	}

	/**
	 * Gets structure for advanced block.
	 *
	 * @param string $moduleName
	 *
	 * @throws AppException
	 *
	 * @return array
	 */
	private function getInvStructure(string $moduleName): array
	{
		$invFields = [];
		foreach (Vtiger_Inventory_Model::getInstance($moduleName)->getFields() as $invField) {
			$invFields['LBL_ADVANCED_BLOCK'][$invField->getColumnName()] = $invField;
		}

		return $invFields;
	}

	/**
	 * Gets structure for related modules.
	 *
	 * @param string $moduleName
	 *
	 * @return array
	 */
	private function getRelatedStructures(string $moduleName): array
	{
		$structure = [];
		foreach (Vtiger_Module_Model::getInstance($moduleName)->getFieldsByReference() as $referenceField) {
			foreach ($referenceField->getReferenceList() as $relatedModuleName) {
				$structure[$relatedModuleName][$referenceField->getName()] = Vtiger_RecordStructure_Model::getInstanceForModule(Vtiger_Module_Model::getInstance($relatedModuleName))->getStructure();
			}
		}
		return $structure;
	}
}
